/**
 * Marlin 3D Printer Firmware
 * Copyright (c) 2020 MarlinFirmware [https://github.com/MarlinFirmware/Marlin]
 *
 * Based on Sprinter and grbl.
 * Copyright (c) 2011 Camiel Gubbels / Erik van der Zalm
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
#pragma once

/**
 * Standard Marlin Status Screen bitmaps
 *
 * Use the Marlin Bitmap Converter to make your own:
 * http://marlinfw.org/tools/u8glib/converter.html
 */

#include "../../inc/MarlinConfig.h"
#include "ultralcd_DOGM.h"

#define BW(N) ((N + 7) / 8)

#if ENABLED(CUSTOM_STATUS_SCREEN_IMAGE) && DISABLED(STATUS_COMBINE_HEATERS)

  #undef STATUS_HEATERS_X
  #undef STATUS_BED_X
  /**
   * Custom _Statusscreen.h files can define:
   * - A custom logo image
   * - A custom heater bitmap
   * - A custom fan bitmap / animation
   *
   * See the included examples for guidance
   */
  #include "../../../_Statusscreen.h"

  #ifdef STATUS_SCREENWIDTH
    #error "Your custom _Statusscreen.h needs to be converted for Marlin 2.0."
  #endif

#endif

//
// Default Status Screen Heater or Hotends bitmaps
//
#if !STATUS_HEATERS_WIDTH && !STATUS_HOTEND1_WIDTH

  #if ENABLED(STATUS_COMBINE_HEATERS)

    #undef STATUS_HOTEND_ANIM
    #undef STATUS_BED_ANIM
    #define STATUS_HEATERS_XSPACE 24

    //
    // Status Screen Combined Heater bitmaps
    //
    #if HAS_HEATED_BED && HOTENDS <= 4

      #if HOTENDS == 0

        #define STATUS_HEATERS_WIDTH 96

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000100,B00010000,B01000000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000010,B00001000,B00100000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000010,B00001000,B00100000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000100,B00010000,B01000000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00001000,B00100000,B10000000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00001000,B00100000,B10000000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000100,B00010000,B01000000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00011111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00011111,B11111111,B11111000
        };

      #elif HOTENDS == 1

        #define STATUS_HEATERS_WIDTH 96

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000100,B00010000,B01000000,
          B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000010,B00001000,B00100000,
          B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000010,B00001000,B00100000,
          B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000100,B00010000,B01000000,
          B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00001000,B00100000,B10000000,
          B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,
          B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,
          B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00001000,B00100000,B10000000,
          B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000100,B00010000,B01000000,
          B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,
          B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00011111,B11111111,B11111000,
          B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00011111,B11111111,B11111000
        };

      #elif HOTENDS == 2

        #define STATUS_HEATERS_WIDTH 96

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000100,B00010000,B01000000,
          B00111110,B11110000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00000010,B00001000,B00100000,
          B00111100,B11110000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00000010,B00001000,B00100000,
          B00111010,B11110000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00000100,B00010000,B01000000,
          B00011110,B11100000,B00000000,B00011110,B11100000,B00000000,B00000000,B00000000,B00000000,B00001000,B00100000,B10000000,
          B00011110,B11100000,B00000000,B00011101,B11100000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,
          B00111110,B11110000,B00000000,B00111011,B11110000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,
          B00111110,B11110000,B00000000,B00111000,B01110000,B00000000,B00000000,B00000000,B00000000,B00001000,B00100000,B10000000,
          B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000100,B00010000,B01000000,
          B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,
          B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B00011111,B11111111,B11111000,
          B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B00011111,B11111111,B11111000
        };

      #elif HOTENDS == 3

        #define STATUS_HEATERS_WIDTH 96

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00000100,B00010000,B01000000,
          B00111110,B11110000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00000010,B00001000,B00100000,
          B00111100,B11110000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00000010,B00001000,B00100000,
          B00111010,B11110000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00000100,B00010000,B01000000,
          B00011110,B11100000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B00001000,B00100000,B10000000,
          B00011110,B11100000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B00010000,B01000001,B00000000,
          B00111110,B11110000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B00010000,B01000001,B00000000,
          B00111110,B11110000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B00001000,B00100000,B10000000,
          B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00000100,B00010000,B01000000,
          B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,
          B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00011111,B11111111,B11111000,
          B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00011111,B11111111,B11111000
        };

      #else // HOTENDS > 3

        #define STATUS_HEATERS_WIDTH 120

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00000100,B00010000,B01000000,
          B00111110,B11110000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00111011,B01110000,B00000000,B00000010,B00001000,B00100000,
          B00111100,B11110000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00000010,B00001000,B00100000,
          B00111010,B11110000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00111011,B01110000,B00000000,B00000100,B00010000,B01000000,
          B00011110,B11100000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B00011011,B01100000,B00000000,B00001000,B00100000,B10000000,
          B00011110,B11100000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B00011000,B00100000,B00000000,B00010000,B01000001,B00000000,
          B00111110,B11110000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B00111111,B01110000,B00000000,B00010000,B01000001,B00000000,
          B00111110,B11110000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B00111111,B01110000,B00000000,B00001000,B00100000,B10000000,
          B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00000100,B00010000,B01000000,
          B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,
          B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00011111,B11111111,B11111000,
          B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00011111,B11111111,B11111000
        };

      #endif // HOTENDS

      #define STATUS_BED_TEXT_X (STATUS_HEATERS_WIDTH - 10)

    #else // !HAS_HEATED_BED || HOTENDS > 3

      #if HOTENDS == 0

        #define STATUS_HEATERS_WIDTH  0

      #elif HOTENDS == 1

        #define STATUS_HEATERS_WIDTH  12

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00011111,B11100000,
          B00011111,B11100000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00001111,B11000000,
          B00000111,B10000000,
          B00000011,B00000000
        };

      #elif HOTENDS == 2

        #define STATUS_HEATERS_WIDTH  36

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,B00000000,B00011111,B11100000,
          B00111110,B11110000,B00000000,B00111100,B11110000,
          B00111100,B11110000,B00000000,B00111011,B01110000,
          B00111010,B11110000,B00000000,B00111111,B01110000,
          B00011110,B11100000,B00000000,B00011110,B11100000,
          B00011110,B11100000,B00000000,B00011101,B11100000,
          B00111110,B11110000,B00000000,B00111011,B11110000,
          B00111110,B11110000,B00000000,B00111000,B01110000,
          B00111111,B11110000,B00000000,B00111111,B11110000,
          B00001111,B11000000,B00000000,B00001111,B11000000,
          B00000111,B10000000,B00000000,B00000111,B10000000,
          B00000011,B00000000,B00000000,B00000011,B00000000
        };

      #elif HOTENDS == 3

        #define STATUS_HEATERS_WIDTH  60

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,
          B00111110,B11110000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,
          B00111100,B11110000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,
          B00111010,B11110000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,
          B00011110,B11100000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,
          B00011110,B11100000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,
          B00111110,B11110000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,
          B00111110,B11110000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,
          B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,
          B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,
          B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,
          B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000
        };

      #elif HOTENDS == 4

        #define STATUS_HEATERS_WIDTH  84

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,
          B00111110,B11110000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00111011,B01110000,
          B00111100,B11110000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,
          B00111010,B11110000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00111011,B01110000,
          B00011110,B11100000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B00011011,B01100000,
          B00011110,B11100000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B00011000,B00100000,
          B00111110,B11110000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B00111111,B01110000,
          B00111110,B11110000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B00111111,B01110000,
          B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,
          B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,
          B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,
          B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000
        };

      #else // HOTENDS > 4

        #define STATUS_HEATERS_WIDTH  108

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,
          B00111110,B11110000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00111011,B01110000,B00000000,B00111000,B01110000,
          B00111100,B11110000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00111011,B11110000,
          B00111010,B11110000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00111011,B01110000,B00000000,B00111000,B11110000,
          B00011110,B11100000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B00011011,B01100000,B00000000,B00011111,B01100000,
          B00011110,B11100000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B00011000,B00100000,B00000000,B00011111,B01100000,
          B00111110,B11110000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B00111111,B01110000,B00000000,B00111011,B01110000,
          B00111110,B11110000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B00111111,B01110000,B00000000,B00111100,B11110000,
          B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,
          B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,
          B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,
          B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000
        };

      #endif // HOTENDS

    #endif // !HAS_HEATED_BED || HOTENDS > 3

  #else // !STATUS_COMBINE_HEATERS

    //
    // Status Screen Hotends bitmaps
    //
    #if HOTENDS

      #define STATUS_HOTEND1_WIDTH  16

      #define MAX_HOTEND_BITMAPS     5
      #if HOTENDS > MAX_HOTEND_BITMAPS
        #define STATUS_HOTEND_BITMAPS MAX_HOTEND_BITMAPS
      #else
        #define STATUS_HOTEND_BITMAPS HOTENDS
      #endif

      #if HOTENDS == 1 || ENABLED(STATUS_HOTEND_NUMBERLESS)

        const unsigned char status_hotend_a_bmp[] PROGMEM = {
          B00011111,B11100000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00011111,B11100000,
          B00011111,B11100000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00001111,B11000000,
          B00000111,B10000000,
          B00000011,B00000000
        };

        #ifdef STATUS_HOTEND_ANIM

          const unsigned char status_hotend_b_bmp[] PROGMEM = {
            B00011111,B11100000,
            B00100000,B00010000,
            B00100000,B00010000,
            B00100000,B00010000,
            B00010000,B00100000,
            B00010000,B00100000,
            B00100000,B00010000,
            B00100000,B00010000,
            B00110000,B00110000,
            B00001000,B01000000,
            B00000100,B10000000,
            B00000011,B00000000
          };

        #endif

      #elif HOTENDS >= 2

        #ifdef STATUS_HOTEND_ANIM

          const unsigned char status_hotend1_a_bmp[] PROGMEM = {
            B00011111,B11100000,
            B00111111,B11110000,
            B00111110,B11110000,
            B00111100,B11110000,
            B00011010,B11100000,
            B00011110,B11100000,
            B00111110,B11110000,
            B00111110,B11110000,
            B00111110,B11110000,
            B00001111,B11000000,
            B00000111,B10000000,
            B00000011,B00000000
          };

          const unsigned char status_hotend1_b_bmp[] PROGMEM = {
            B00011111,B11100000,
            B00100000,B00010000,
            B00100001,B00010000,
            B00100011,B00010000,
            B00010101,B00100000,
            B00010001,B00100000,
            B00100001,B00010000,
            B00100001,B00010000,
            B00110001,B00110000,
            B00001000,B01000000,
            B00000100,B10000000,
            B00000011,B00000000
          };

          const unsigned char status_hotend2_a_bmp[] PROGMEM = {
            B00011111,B11100000,
            B00111111,B11110000,
            B00111100,B11110000,
            B00111011,B01110000,
            B00011111,B01100000,
            B00011110,B11100000,
            B00111101,B11110000,
            B00111011,B11110000,
            B00111000,B01110000,
            B00001111,B11000000,
            B00000111,B10000000,
            B00000011,B00000000
          };

          const unsigned char status_hotend2_b_bmp[] PROGMEM = {
            B00011111,B11100000,
            B00100000,B00010000,
            B00100011,B00010000,
            B00100100,B10010000,
            B00010000,B10100000,
            B00010001,B00100000,
            B00100010,B00010000,
            B00100100,B00010000,
            B00110111,B10110000,
            B00001000,B01000000,
            B00000100,B10000000,
            B00000011,B00000000
          };

        #else

          const unsigned char status_hotend1_a_bmp[] PROGMEM = {
            B00011111,B11100000,
            B00111110,B11110000,
            B00111100,B11110000,
            B00111010,B11110000,
            B00011110,B11100000,
            B00011110,B11100000,
            B00111110,B11110000,
            B00111110,B11110000,
            B00111111,B11110000,
            B00001111,B11000000,
            B00000111,B10000000,
            B00000011,B00000000
          };

          const unsigned char status_hotend2_a_bmp[] PROGMEM = {
            B00011111,B11100000,
            B00111100,B11110000,
            B00111011,B01110000,
            B00111111,B01110000,
            B00011110,B11100000,
            B00011101,B11100000,
            B00111011,B11110000,
            B00111000,B01110000,
            B00111111,B11110000,
            B00001111,B11000000,
            B00000111,B10000000,
            B00000011,B00000000
          };

        #endif

        #if STATUS_HOTEND_BITMAPS >= 3

          #ifdef STATUS_HOTEND_ANIM

            const unsigned char status_hotend3_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111111,B11110000,
              B00111100,B11110000,
              B00111011,B01110000,
              B00011111,B01100000,
              B00011100,B11100000,
              B00111111,B01110000,
              B00111011,B01110000,
              B00111100,B11110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

            const unsigned char status_hotend3_b_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00100000,B00010000,
              B00100011,B00010000,
              B00100100,B10010000,
              B00010000,B10100000,
              B00010011,B00100000,
              B00100000,B10010000,
              B00100100,B10010000,
              B00110011,B00110000,
              B00001000,B01000000,
              B00000100,B10000000,
              B00000011,B00000000
            };

          #else

            const unsigned char status_hotend3_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111100,B11110000,
              B00111011,B01110000,
              B00111111,B01110000,
              B00011100,B11100000,
              B00011111,B01100000,
              B00111011,B01110000,
              B00111100,B11110000,
              B00111111,B11110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

          #endif

        #endif

        #if STATUS_HOTEND_BITMAPS >= 4

          #ifdef STATUS_HOTEND_ANIM

            const unsigned char status_hotend4_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111111,B11110000,
              B00111011,B01110000,
              B00111011,B01110000,
              B00011011,B01100000,
              B00011011,B01100000,
              B00111000,B00110000,
              B00111111,B01110000,
              B00111111,B01110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

            const unsigned char status_hotend4_b_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00100000,B00010000,
              B00100100,B10010000,
              B00100100,B10010000,
              B00010100,B10100000,
              B00010100,B10100000,
              B00100111,B11010000,
              B00100000,B10010000,
              B00110000,B10110000,
              B00001000,B01000000,
              B00000100,B10000000,
              B00000011,B00000000
            };

          #else

            const unsigned char status_hotend4_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111011,B01110000,
              B00111011,B01110000,
              B00111011,B01110000,
              B00011011,B01100000,
              B00011000,B00100000,
              B00111111,B01110000,
              B00111111,B01110000,
              B00111111,B11110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

          #endif

        #endif

        #if STATUS_HOTEND_BITMAPS >= 5

          #ifdef STATUS_HOTEND_ANIM

            const unsigned char status_hotend5_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111111,B11110000,
              B00111000,B01110000,
              B00111011,B11110000,
              B00011000,B11100000,
              B00011111,B01100000,
              B00111111,B01110000,
              B00111011,B01110000,
              B00111100,B11110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

            const unsigned char status_hotend5_b_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00100000,B00010000,
              B00100111,B10010000,
              B00100100,B00010000,
              B00010111,B00100000,
              B00010000,B10100000,
              B00100000,B10010000,
              B00100100,B10010000,
              B00110011,B00110000,
              B00001000,B01000000,
              B00000100,B10000000,
              B00000011,B00000000
            };

          #else

            const unsigned char status_hotend5_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111000,B01110000,
              B00111011,B11110000,
              B00111000,B11110000,
              B00011111,B01100000,
              B00011111,B01100000,
              B00111011,B01110000,
              B00111100,B11110000,
              B00111111,B11110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

          #endif

        #endif

      #endif

    #else
      #define STATUS_HEATERS_HEIGHT 20
    #endif

  #endif

#endif // !STATUS_HEATERS_WIDTH && !STATUS_HOTEND1_WIDTH

// LASER / SPINDLE
#if !STATUS_CUTTER_WIDTH && HAS_CUTTER
  #define STATUS_CUTTER_WIDTH 24
  #define STATUS_CUTTER_X 80
  #if ENABLED(LASER_FEATURE)
    #ifdef STATUS_CUTTER_ANIM
      const unsigned char status_cutter_on_bmp[] PROGMEM = {
        B00000000,B00100100,B00000000,
        B00000000,B01100110,B00000000,
        B00000000,B11000011,B00000000,
        B00000001,B10011001,B10000000,
        B00000011,B00100100,B11000000,
        B00000000,B01000010,B00000000,
        B00000000,B01000010,B00000000,
        B00000011,B00100100,B11000000,
        B00000001,B10011001,B10000000,
        B00000000,B11000011,B00000000,
        B00000000,B01100110,B00000000,
        B00000000,B00100100,B00000000
      };
      const unsigned char status_cutter_bmp[] PROGMEM = {
        B00000000,B00100100,B00000000,
        B00000000,B01100110,B00000000,
        B00000000,B00000000,B00000000,
        B00000001,B00000000,B10000000,
        B00000011,B00000000,B11000000,
        B00000000,B00011000,B00000000,
        B00000000,B00011000,B00000000,
        B00000011,B00000000,B11000000,
        B00000001,B00000000,B10000000,
        B00000000,B00000000,B00000000,
        B00000000,B01100110,B00000000,
        B00000000,B00100100,B00000000
      };
    #else
      const unsigned char status_cutter_bmp[] PROGMEM = {
        B00000000,B00100100,B00000000,
        B00000000,B01100110,B00000000,
        B00000000,B11000011,B00000000,
        B00000001,B10000001,B10000000,
        B00000011,B00000000,B11000000,
        B00000000,B00000000,B00000000,
        B00000000,B00000000,B00000000,
        B00000011,B00000000,B11000000,
        B00000001,B10000001,B10000000,
        B00000000,B11000011,B00000000,
        B00000000,B01100110,B00000000,
        B00000000,B00100100,B00000000
      };
    #endif
  #else
    #ifdef STATUS_CUTTER_ANIM
      const unsigned char status_cutter_on_bmp[] PROGMEM = {
        B00000001,B11111110,B10000000,
        B00000000,B11000000,B00000000,
        B00000001,B10000000,B10000000,
        B00000001,B00000000,B10000000,
        B00000001,B11111100,B10000000,
        B00000000,B11100000,B00000000,
        B00000001,B11000000,B10000000,
        B00000000,B10000001,B00000000,
        B00000000,B01111010,B00000000,
        B00000000,B00110100,B00000000,
        B00000000,B00011000,B00000000,
        B00000000,B00000000,B00000000
      };
      const unsigned char status_cutter_bmp[] PROGMEM = {
        B00000001,B11111110,B10000000,
        B00000000,B11000000,B00000000,
        B00000001,B10000000,B10000000,
        B00000001,B00000000,B10000000,
        B00000001,B11111100,B10000000,
        B00000000,B11100000,B00000000,
        B00000001,B11000000,B10000000,
        B00000000,B10000001,B00000000,
        B00000000,B01111010,B00000000,
        B00000000,B00110100,B00000000,
        B00000000,B00011000,B00000000,
        B00000000,B00000000,B00000000
      };
    #else
      const unsigned char status_cutter_bmp[] PROGMEM = {
        B00000001,B11000010,B10000000,
        B00000001,B00011100,B10000000,
        B00000000,B11100001,B00000000,
        B00000001,B00001110,B10000000,
        B00000001,B01110000,B10000000,
        B00000000,B10000111,B10000000,
        B00000001,B00111111,B10000000,
        B00000000,B11111111,B00000000,
        B00000000,B01111110,B00000000,
        B00000000,B00111100,B00000000,
        B00000000,B00011000,B00000000,
        B00000000,B00000000,B00000000
      };
    #endif
  #endif
#endif // LASER / SPINDLE

//
// Default Status Screen Bed bitmaps
//
#if !STATUS_BED_WIDTH && HAS_HEATED_BED && DISABLED(STATUS_COMBINE_HEATERS)

  #if ENABLED(STATUS_ALT_BED_BITMAP)

    #define STATUS_BED_ANIM
    #define STATUS_BED_WIDTH  24
    #ifndef STATUS_BED_X
      #define STATUS_BED_X   (LCD_PIXEL_WIDTH - (STATUS_BED_BYTEWIDTH + STATUS_CHAMBER_BYTEWIDTH + STATUS_FAN_BYTEWIDTH) * 8)
    #endif
    #define STATUS_BED_TEXT_X (STATUS_BED_X + 11)

    const unsigned char status_bed_bmp[] PROGMEM = {
      B11111111,B11111111,B11000000,
      B01000000,B00000000,B00100000,
      B00100000,B00000000,B00010000,
      B00010000,B00000000,B00001000,
      B00001000,B00000000,B00000100,
      B00000100,B00000000,B00000010,
      B00000011,B11111111,B11111111
    };

    const unsigned char status_bed_on_bmp[] PROGMEM = {
      B00000010,B00100010,B00000000,
      B00000100,B01000100,B00000000,
      B00000100,B01000100,B00000000,
      B00000010,B00100010,B00000000,
      B00000001,B00010001,B00000000,
      B11111111,B11111111,B11000000,
      B01000000,B10001000,B10100000,
      B00100001,B00010001,B00010000,
      B00010010,B00100010,B00001000,
      B00001000,B00000000,B00000100,
      B00000100,B00000000,B00000010,
      B00000011,B11111111,B11111111
    };

  #else

    #define STATUS_BED_WIDTH  21
    #ifndef STATUS_BED_X
      #define STATUS_BED_X   (LCD_PIXEL_WIDTH - (STATUS_BED_BYTEWIDTH + STATUS_CHAMBER_BYTEWIDTH + STATUS_FAN_BYTEWIDTH) * 8)
    #endif

    #ifdef STATUS_BED_ANIM

      const unsigned char status_bed_bmp[] PROGMEM = {
        B00011111,B11111111,B11111000,
        B00011111,B11111111,B11111000
      };

      const unsigned char status_bed_on_bmp[] PROGMEM = {
        B00000100,B00010000,B01000000,
        B00000010,B00001000,B00100000,
        B00000010,B00001000,B00100000,
        B00000100,B00010000,B01000000,
        B00001000,B00100000,B10000000,
        B00010000,B01000001,B00000000,
        B00010000,B01000001,B00000000,
        B00001000,B00100000,B10000000,
        B00000100,B00010000,B01000000,
        B00000000,B00000000,B00000000,
        B00011111,B11111111,B11111000,
        B00011111,B11111111,B11111000
      };

    #else

      const unsigned char status_bed_bmp[] PROGMEM = {
        B00000100,B00010000,B01000000,
        B00000010,B00001000,B00100000,
        B00000010,B00001000,B00100000,
        B00000100,B00010000,B01000000,
        B00001000,B00100000,B10000000,
        B00010000,B01000001,B00000000,
        B00010000,B01000001,B00000000,
        B00001000,B00100000,B10000000,
        B00000100,B00010000,B01000000,
        B00000000,B00000000,B00000000,
        B00011111,B11111111,B11111000,
        B00011111,B11111111,B11111000
      };

    #endif

  #endif
#endif

#ifndef STATUS_BED_WIDTH
  #define STATUS_BED_WIDTH 0
#endif

#if !STATUS_CHAMBER_WIDTH && HAS_TEMP_CHAMBER && ((HOTENDS <= 4 && !HAS_HEATED_BED) || (HOTENDS <= 3 && HAS_HEATED_BED))
  #define STATUS_CHAMBER_WIDTH 21
  #if STATUS_HEATERS_WIDTH
    #if ENABLED(STATUS_COMBINE_HEATERS)
      #define STATUS_CHAMBER_X (LCD_PIXEL_WIDTH - 2 - (STATUS_CHAMBER_BYTEWIDTH) * 8)
    #elif HAS_FAN0 && HAS_HEATED_BED && HOTENDS <= 2
      #define STATUS_CHAMBER_X (LCD_PIXEL_WIDTH - 2 - (STATUS_HEATERS_BYTEWIDTH - STATUS_CHAMBER_BYTEWIDTH) * 8)
    #elif HAS_FAN0 && !HAS_HEATED_BED
      #define STATUS_CHAMBER_X (LCD_PIXEL_WIDTH - (STATUS_CHAMBER_BYTEWIDTH + STATUS_FAN_BYTEWIDTH) * 8)
    #else
      #define STATUS_CHAMBER_X (LCD_PIXEL_WIDTH - (STATUS_CHAMBER_BYTEWIDTH) * 8)
    #endif
  #endif

  #ifdef STATUS_CHAMBER_ANIM

    const unsigned char status_chamber_bmp[] PROGMEM = {
      B00011111,B11111111,B11111000,
      B00010000,B00000000,B00001000,
      B00010000,B00000000,B00001000,
      B00010000,B00000000,B00001000,
      B00010000,B00000000,B00001000,
      B00010000,B00000000,B00001000,
      B00010000,B00000000,B00001000,
      B00010000,B00000000,B00001000,
      B00010000,B00000000,B00001000,
      B00010000,B00000000,B00001000,
      B00011111,B11111111,B11111000,
      B00011111,B11111111,B11111000
    };
    const unsigned char status_chamber_on_bmp[] PROGMEM = {
      B00011111,B11111111,B11111000,
      B00010000,B00000000,B00001000,
      B00010000,B10000100,B00001000,
      B00010000,B01000010,B00001000,
      B00010000,B01000010,B00001000,
      B00010000,B10000100,B00001000,
      B00010001,B00001000,B00001000,
      B00010001,B00001000,B00001000,
      B00010000,B10000100,B00001000,
      B00010000,B00000000,B00001000,
      B00011111,B11111111,B11111000,
      B00011111,B11111111,B11111000
    };

  #else

    const unsigned char status_chamber_bmp[] PROGMEM = {
      B00011111,B11111111,B11111000,
      B00010000,B00000000,B00001000,
      B00010000,B10000100,B00001000,
      B00010000,B01000010,B00001000,
      B00010000,B01000010,B00001000,
      B00010000,B10000100,B00001000,
      B00010001,B00001000,B00001000,
      B00010001,B00001000,B00001000,
      B00010000,B10000100,B00001000,
      B00010000,B00000000,B00001000,
      B00011111,B11111111,B11111000,
      B00011111,B11111111,B11111000
    };

  #endif
#else // HAS_HEATED_CHAMBER
  #define STATUS_CHAMBER_WIDTH 0
#endif

#define BED_OR_CHAMBER_OR_FAN (BED_OR_CHAMBER || HAS_FAN0)

// Can also be overridden in Configuration_adv.h
// If you can afford it, try the 3-frame fan animation!
// Don't compile in the fan animation with no fan
#if !HAS_FAN0 || (HOTENDS == 5 || (HOTENDS == 4 && BED_OR_CHAMBER) || (ENABLED(STATUS_COMBINE_HEATERS) && HAS_HEATED_CHAMBER))
  #undef STATUS_FAN_FRAMES
#elif !STATUS_FAN_FRAMES
  #define STATUS_FAN_FRAMES 2
#elif STATUS_FAN_FRAMES > 4
  #error "A maximum of 4 fan animation frames is currently supported."
#endif

#if HOTENDS > 4
  #undef STATUS_LOGO_WIDTH
  #undef STATUS_HEATERS_XSPACE
  #define STATUS_HEATERS_XSPACE 24
#endif

//
// Provide default Fan Bitmaps
//
#if !STATUS_FAN_WIDTH && STATUS_FAN_FRAMES > 0

  // Provide a fan animation if none exists

  #if STATUS_FAN_FRAMES <= 2

    #define STATUS_FAN_Y      2
    #define STATUS_FAN_WIDTH 20

    #if ENABLED(STATUS_ALT_FAN_BITMAP)

      const unsigned char status_fan0_bmp[] PROGMEM = {
        B00000001,B11111110,B00000000,
        B00000110,B00000001,B10000000,
        B00001000,B11111100,B01000000,
        B00010000,B11111100,B00100000,
        B00010000,B01111000,B00100000,
        B00100000,B00110000,B00010000,
        B00101100,B00000000,B11010000,
        B00101110,B00110001,B11010000,
        B00101111,B01111011,B11010000,
        B00101111,B01111011,B11010000,
        B00101110,B00110001,B11010000,
        B00101100,B00000000,B11010000,
        B00100000,B00110000,B00010000,
        B00010000,B01111000,B00100000,
        B00010000,B11111100,B00100000,
        B00001000,B11111100,B01000000,
        B00000110,B00000001,B10000000,
        B00000001,B11111110,B00000000
      };

      #if STATUS_FAN_FRAMES == 2
        const unsigned char status_fan1_bmp[] PROGMEM = {
          B00000001,B11111110,B00000000,
          B00000110,B00000001,B10000000,
          B00001001,B10000110,B01000000,
          B00010011,B10000111,B00100000,
          B00010111,B10000111,B10100000,
          B00101111,B10000111,B11010000,
          B00101111,B00000011,B11010000,
          B00100000,B00110000,B00010000,
          B00100000,B01111000,B00010000,
          B00100000,B01111000,B00010000,
          B00100000,B00110000,B00010000,
          B00101111,B00000011,B11010000,
          B00101111,B10000111,B11010000,
          B00010111,B10000111,B10100000,
          B00010011,B10000111,B00100000,
          B00001001,B10000110,B01000000,
          B00000110,B00000001,B10000000,
          B00000001,B11111110,B00000000
        };
      #endif

    #else // !STATUS_ALT_FAN_BITMAP

      const unsigned char status_fan0_bmp[] PROGMEM = {
        B00111111,B11111111,B11110000,
        B00111000,B00000000,B01110000,
        B00110000,B11111100,B00110000,
        B00100000,B11111100,B00010000,
        B00100000,B01111000,B00010000,
        B00100000,B00110000,B00010000,
        B00101100,B00000000,B11010000,
        B00101110,B00110001,B11010000,
        B00101111,B01111011,B11010000,
        B00101111,B01111011,B11010000,
        B00101110,B00110001,B11010000,
        B00101100,B00000000,B11010000,
        B00100000,B00110000,B00010000,
        B00100000,B01111000,B00010000,
        B00100000,B11111100,B00010000,
        B00110000,B11111100,B00110000,
        B00111000,B00000000,B01110000,
        B00111111,B11111111,B11110000
      };

      #if STATUS_FAN_FRAMES == 2
        const unsigned char status_fan1_bmp[] PROGMEM = {
          B00111111,B11111111,B11110000,
          B00111000,B00000000,B01110000,
          B00110001,B10000110,B00110000,
          B00100011,B10000111,B00010000,
          B00100111,B10000111,B10010000,
          B00101111,B10000111,B11010000,
          B00101111,B00000011,B11010000,
          B00100000,B00110000,B00010000,
          B00100000,B01111000,B00010000,
          B00100000,B01111000,B00010000,
          B00100000,B00110000,B00010000,
          B00101111,B00000011,B11010000,
          B00101111,B10000111,B11010000,
          B00100111,B10000111,B10010000,
          B00100011,B10000111,B00010000,
          B00110001,B10000110,B00110000,
          B00111000,B00000000,B01110000,
          B00111111,B11111111,B11110000
        };
      #endif

    #endif // !STATUS_ALT_FAN_BITMAP

  #elif STATUS_FAN_FRAMES == 3

    #define STATUS_FAN_WIDTH 20

    #if ENABLED(STATUS_ALT_FAN_BITMAP)

      const unsigned char status_fan0_bmp[] PROGMEM = {
        B00000001,B11111111,B00000000,
        B00000110,B00000000,B11000000,
        B00001001,B00000001,B00100000,
        B00010111,B10000011,B11010000,
        B00010111,B10000011,B11010000,
        B00101111,B11000111,B11101000,
        B00100111,B11000111,B11001000,
        B00100001,B11111111,B00001000,
        B00100000,B01111100,B00001000,
        B00100000,B01111100,B00001000,
        B00100000,B01111100,B00001000,
        B00100001,B11111111,B00001000,
        B00100111,B11000111,B11001000,
        B00101111,B11000111,B11101000,
        B00010111,B10000011,B11010000,
        B00010111,B10000011,B11010000,
        B00001001,B00000001,B00100000,
        B00000110,B00000000,B11000000,
        B00000001,B11111111,B00000000
      };
      const unsigned char status_fan1_bmp[] PROGMEM = {
        B00000001,B11111111,B00000000,
        B00000110,B00110000,B11000000,
        B00001001,B11110000,B00100000,
        B00010001,B11110000,B00010000,
        B00010000,B11110000,B00010000,
        B00100000,B11110000,B01101000,
        B00100000,B00110001,B11101000,
        B00100000,B00111001,B11101000,
        B00100000,B01111111,B11111000,
        B00111111,B11111111,B11111000,
        B00111111,B11111100,B00001000,
        B00101111,B00111000,B00001000,
        B00101110,B00011000,B00001000,
        B00101100,B00011110,B00001000,
        B00010000,B00011110,B00010000,
        B00010000,B00011111,B00010000,
        B00001000,B00011111,B00100000,
        B00000110,B00011000,B11000000,
        B00000001,B11111111,B00000000
      };
      const unsigned char status_fan2_bmp[] PROGMEM = {
        B00000001,B11111111,B00000000,
        B00000110,B00011000,B11000000,
        B00001000,B00011111,B00100000,
        B00010000,B00011111,B10010000,
        B00010100,B00011111,B00010000,
        B00101110,B00011110,B00001000,
        B00101111,B00011100,B00001000,
        B00101111,B10111000,B00001000,
        B00111111,B11111100,B00001000,
        B00111111,B11111111,B11111000,
        B00100000,B01111111,B11111000,
        B00100000,B00111011,B11101000,
        B00100000,B01110001,B11101000,
        B00100000,B11110000,B11101000,
        B00010001,B11110000,B01010000,
        B00010011,B11110000,B00010000,
        B00001001,B11110000,B00100000,
        B00000110,B00110000,B11000000,
        B00000001,B11111111,B00000000
      };

    #else // !STATUS_ALT_FAN_BITMAP

      const unsigned char status_fan0_bmp[] PROGMEM = {
        B00111111,B11111111,B11111000,
        B00111110,B00000000,B11111000,
        B00111001,B00000001,B00111000,
        B00110111,B10000011,B11011000,
        B00110111,B10000011,B11011000,
        B00101111,B11000111,B11101000,
        B00100111,B11000111,B11001000,
        B00100001,B11111111,B00001000,
        B00100000,B01111100,B00001000,
        B00100000,B01111100,B00001000,
        B00100000,B01111100,B00001000,
        B00100001,B11111111,B00001000,
        B00100111,B11000111,B11001000,
        B00101111,B11000111,B11101000,
        B00110111,B10000011,B11011000,
        B00110111,B10000011,B11011000,
        B00111001,B00000001,B00111000,
        B00111110,B00000000,B11111000,
        B00111111,B11111111,B11111000
      };
      const unsigned char status_fan1_bmp[] PROGMEM = {
        B00111111,B11111111,B11111000,
        B00111110,B00110000,B11111000,
        B00111001,B11110000,B00111000,
        B00110001,B11110000,B00011000,
        B00110000,B11110000,B00011000,
        B00100000,B11110000,B01101000,
        B00100000,B00110001,B11101000,
        B00100000,B00111001,B11101000,
        B00100000,B01111111,B11111000,
        B00111111,B11111111,B11111000,
        B00111111,B11111100,B00001000,
        B00101111,B00111000,B00001000,
        B00101110,B00011000,B00001000,
        B00101100,B00011110,B00001000,
        B00110000,B00011110,B00011000,
        B00110000,B00011111,B00011000,
        B00111000,B00011111,B00111000,
        B00111110,B00011000,B11111000,
        B00111111,B11111111,B11111000
      };
      const unsigned char status_fan2_bmp[] PROGMEM = {
        B00111111,B11111111,B11111000,
        B00111110,B00011000,B11111000,
        B00111000,B00011111,B00111000,
        B00110000,B00011111,B10011000,
        B00110100,B00011111,B00011000,
        B00101110,B00011110,B00001000,
        B00101111,B00011100,B00001000,
        B00101111,B10111000,B00001000,
        B00111111,B11111100,B00001000,
        B00111111,B11111111,B11111000,
        B00100000,B01111111,B11111000,
        B00100000,B00111011,B11101000,
        B00100000,B01110001,B11101000,
        B00100000,B11110000,B11101000,
        B00110001,B11110000,B01011000,
        B00110011,B11110000,B00011000,
        B00111001,B11110000,B00111000,
        B00111110,B00110000,B11111000,
        B00111111,B11111111,B11111000
      };

    #endif // !STATUS_ALT_FAN_BITMAP

  #elif STATUS_FAN_FRAMES == 4

    #define STATUS_FAN_WIDTH 20

    #if ENABLED(STATUS_ALT_FAN_BITMAP)

      const unsigned char status_fan0_bmp[] PROGMEM = {
        B00000001,B11111111,B00000000,
        B00000110,B00000000,B11000000,
        B00001000,B00111111,B00100000,
        B00010000,B01111110,B00010000,
        B00010000,B01111100,B00010000,
        B00101000,B01111100,B00001000,
        B00101100,B00111000,B00001000,
        B00101111,B00111001,B11001000,
        B00101111,B11111111,B11101000,
        B00101111,B11000111,B11101000,
        B00101111,B11111111,B11101000,
        B00100111,B00111001,B11101000,
        B00100000,B00111000,B01101000,
        B00100000,B01111100,B00101000,
        B00010000,B01111100,B00010000,
        B00010000,B11111100,B00010000,
        B00001001,B11111000,B00100000,
        B00000110,B00000000,B11000000,
        B00000001,B11111111,B00000000
      };
      const unsigned char status_fan1_bmp[] PROGMEM = {
        B00000001,B11111111,B00000000,
        B00000110,B00000000,B11000000,
        B00001000,B00001111,B00100000,
        B00010100,B00011111,B11010000,
        B00010110,B00011111,B10010000,
        B00101111,B00011111,B00001000,
        B00101111,B10011110,B00001000,
        B00101111,B11111100,B00001000,
        B00101111,B11011100,B00001000,
        B00100111,B11101111,B11001000,
        B00100000,B01110111,B11101000,
        B00100000,B01111111,B11101000,
        B00100000,B11110011,B11101000,
        B00100001,B11110001,B11101000,
        B00010011,B11110000,B11010000,
        B00010111,B11110000,B01010000,
        B00001001,B11100000,B00100000,
        B00000110,B00000000,B11000000,
        B00000001,B11111111,B00000000
      };
      const unsigned char status_fan2_bmp[] PROGMEM = {
        B00000001,B11111111,B00000000,
        B00000110,B10000000,B11000000,
        B00001001,B10000000,B00100000,
        B00010111,B10000001,B11010000,
        B00010111,B11000011,B11010000,
        B00100111,B11000111,B11101000,
        B00100011,B11000111,B11111000,
        B00100001,B11111111,B10001000,
        B00100000,B01101100,B00001000,
        B00100000,B01101100,B00001000,
        B00100000,B01101100,B00001000,
        B00100011,B11111111,B00001000,
        B00111111,B11000111,B10001000,
        B00101111,B11000111,B11001000,
        B00010111,B10000111,B11010000,
        B00010111,B00000011,B11010000,
        B00001000,B00000011,B00100000,
        B00000110,B00000010,B11000000,
        B00000001,B11111111,B00000000
      };
      const unsigned char status_fan3_bmp[] PROGMEM = {
        B00000001,B11111111,B00000000,
        B00000110,B00000000,B11000000,
        B00001001,B11110000,B00100000,
        B00010001,B11100000,B00010000,
        B00010001,B11100000,B00010000,
        B00100001,B11100001,B11101000,
        B00100000,B11110011,B11101000,
        B00100000,B01111111,B11101000,
        B00100000,B01110111,B11101000,
        B00101000,B11101110,B00101000,
        B00101111,B11011100,B00001000,
        B00101111,B11111100,B00001000,
        B00101111,B10011110,B00001000,
        B00101111,B00001111,B00001000,
        B00010000,B00001111,B00010000,
        B00010000,B00001111,B00010000,
        B00001000,B00011111,B00100000,
        B00000110,B00000000,B11000000,
        B00000001,B11111111,B00000000
      };

    #else // !STATUS_ALT_FAN_BITMAP

      const unsigned char status_fan0_bmp[] PROGMEM = {
        B00111111,B11111111,B11111000,
        B00111110,B00000000,B11111000,
        B00111000,B00111111,B00111000,
        B00110000,B01111110,B00011000,
        B00110000,B01111100,B00011000,
        B00101000,B01111100,B00001000,
        B00101100,B00111000,B00001000,
        B00101111,B00111001,B11001000,
        B00101111,B11111111,B11101000,
        B00101111,B11000111,B11101000,
        B00101111,B11111111,B11101000,
        B00100111,B00111001,B11101000,
        B00100000,B00111000,B01101000,
        B00100000,B01111100,B00101000,
        B00110000,B01111100,B00011000,
        B00110000,B11111100,B00011000,
        B00111001,B11111000,B00111000,
        B00111110,B00000000,B11111000,
        B00111111,B11111111,B11111000
      };
      const unsigned char status_fan1_bmp[] PROGMEM = {
        B00111111,B11111111,B11111000,
        B00111110,B00000000,B11111000,
        B00111000,B00001111,B00111000,
        B00110100,B00011111,B11011000,
        B00110110,B00011111,B10011000,
        B00101111,B00011111,B00001000,
        B00101111,B10011110,B00001000,
        B00101111,B11111100,B00001000,
        B00101111,B11011100,B00001000,
        B00100111,B11101111,B11001000,
        B00100000,B01110111,B11101000,
        B00100000,B01111111,B11101000,
        B00100000,B11110011,B11101000,
        B00100001,B11110001,B11101000,
        B00110011,B11110000,B11011000,
        B00110111,B11110000,B01011000,
        B00111001,B11100000,B00111000,
        B00111110,B00000000,B11111000,
        B00111111,B11111111,B11111000
      };
      const unsigned char status_fan2_bmp[] PROGMEM = {
        B00111111,B11111111,B11111000,
        B00111110,B10000000,B11111000,
        B00111001,B10000000,B00111000,
        B00110111,B10000001,B11011000,
        B00110111,B11000011,B11011000,
        B00100111,B11000111,B11101000,
        B00100011,B11000111,B11111000,
        B00100001,B11111111,B10001000,
        B00100000,B01101100,B00001000,
        B00100000,B01101100,B00001000,
        B00100000,B01101100,B00001000,
        B00100011,B11111111,B00001000,
        B00111111,B11000111,B10001000,
        B00101111,B11000111,B11001000,
        B00110111,B10000111,B11011000,
        B00110111,B00000011,B11011000,
        B00111000,B00000011,B00111000,
        B00111110,B00000010,B11111000,
        B00111111,B11111111,B11111000
      };
      const unsigned char status_fan3_bmp[] PROGMEM = {
        B00111111,B11111111,B11111000,
        B00111110,B00000000,B11111000,
        B00111001,B11110000,B00111000,
        B00110001,B11100000,B00011000,
        B00110001,B11100000,B00011000,
        B00100001,B11100001,B11101000,
        B00100000,B11110011,B11101000,
        B00100000,B01111111,B11101000,
        B00100000,B01110111,B11101000,
        B00101000,B11101110,B00101000,
        B00101111,B11011100,B00001000,
        B00101111,B11111100,B00001000,
        B00101111,B10011110,B00001000,
        B00101111,B00001111,B00001000,
        B00110000,B00001111,B00011000,
        B00110000,B00001111,B00011000,
        B00111000,B00011111,B00111000,
        B00111110,B00000000,B11111000,
        B00111111,B11111111,B11111000
      };

    #endif // !STATUS_ALT_FAN_BITMAP

  #endif
#else
  #undef STATUS_FAN_FRAMES
  #define STATUS_FAN_WIDTH 0
#endif

#define _EXTRA_WIDTH (STATUS_FAN_WIDTH + STATUS_CHAMBER_WIDTH + STATUS_BED_WIDTH)

//
// Heater Bitmap X Space Requirements
//
#if !defined(STATUS_HEATERS_XSPACE) && (STATUS_HOTEND1_WIDTH || STATUS_HEATERS_WIDTH)
  #if (HOTENDS == 3 || HOTENDS == 4) && ENABLED(STATUS_COMBINE_HEATERS)
    // If more heaters or they're combined, 3 bytes
    #define STATUS_HEATERS_XSPACE 24
  #elif STATUS_LOGO_WIDTH > (LCD_PIXEL_WIDTH - (_EXTRA_WIDTH) - 26 * (HOTENDS)) // 128 - (20 + 24 + 26) == 58
    // If the logo won't fit at 26 width
    #define STATUS_HEATERS_XSPACE 24
  #else
    #define STATUS_HEATERS_XSPACE 26
  #endif
#endif

#if ENABLED(CUSTOM_STATUS_SCREEN_IMAGE)

  //
  // Disable the logo bitmap if insufficient space
  //
  #if STATUS_HEATERS_XSPACE
    #define _HEATERS_WIDTH (HOTENDS * (STATUS_HEATERS_XSPACE)) // as many hotends as possible
  #elif STATUS_HEATERS_WIDTH
    #define _HEATERS_WIDTH STATUS_HEATERS_WIDTH
  #elif HOTENDS
    #error "Status screen heaters region was not specified."
  #endif
  #if STATUS_LOGO_WIDTH > (LCD_PIXEL_WIDTH - (_EXTRA_WIDTH + _HEATERS_WIDTH))
    #warning "Unable to fit custom Status Screen logo. Disabling."
    #undef STATUS_LOGO_WIDTH
  #endif

  #if (HOTENDS > 1 && STATUS_LOGO_WIDTH && BED_OR_CHAMBER_OR_FAN) || (HOTENDS >= 3 && !BED_OR_CHAMBER_OR_FAN)
    #define _STATUS_HEATERS_X(H,S,N) ((LCD_PIXEL_WIDTH - (H * (S + N)) - (_EXTRA_WIDTH) + (STATUS_LOGO_WIDTH)) / 2)
    #if STATUS_HOTEND1_WIDTH
      #if HOTENDS > 2
        #define STATUS_HEATERS_X _STATUS_HEATERS_X(HOTENDS, STATUS_HOTEND1_WIDTH, 6)
      #else
        #define STATUS_HEATERS_X _STATUS_HEATERS_X(HOTENDS, STATUS_HOTEND1_WIDTH, 4)
      #endif
    #else
      #define STATUS_HEATERS_X _STATUS_HEATERS_X(1, STATUS_HEATERS_WIDTH, 4)
    #endif
  #endif

#endif

//
// Custom Logo Bitmap Properties
//
#ifndef STATUS_LOGO_WIDTH
  #define STATUS_LOGO_WIDTH 0
#endif
#ifndef STATUS_LOGO_BYTEWIDTH
  #define STATUS_LOGO_BYTEWIDTH BW(STATUS_LOGO_WIDTH)
#endif
#if STATUS_LOGO_WIDTH
  #ifndef STATUS_LOGO_X
    #define STATUS_LOGO_X 0
  #endif
  #ifndef STATUS_LOGO_Y
    #define STATUS_LOGO_Y _MIN(0U, (10 - (STATUS_LOGO_HEIGHT) / 2))
  #endif
  #ifndef STATUS_LOGO_HEIGHT
    #define STATUS_LOGO_HEIGHT (sizeof(status_logo_bmp) / (STATUS_LOGO_BYTEWIDTH))
  #endif
  static_assert(
    sizeof(status_logo_bmp) == (STATUS_LOGO_BYTEWIDTH) * (STATUS_LOGO_HEIGHT),
    "Status logo bitmap (status_logo_bmp) dimensions don't match data."
  );
#endif

//
// Hotend Heater Bitmap starting X position
//
#if !defined(STATUS_HEATERS_X) && (STATUS_HOTEND1_WIDTH || STATUS_HEATERS_WIDTH)
  #if STATUS_LOGO_BYTEWIDTH
    #define STATUS_HEATERS_X (STATUS_LOGO_BYTEWIDTH * 8)
  #elif ((STATUS_CHAMBER_WIDTH || STATUS_FAN_WIDTH) && (STATUS_BED_WIDTH  && STATUS_HOTEND_BITMAPS == 3)) || \
        ((STATUS_CHAMBER_WIDTH || STATUS_FAN_WIDTH  ||  STATUS_BED_WIDTH) && STATUS_HOTEND_BITMAPS == 4)
    #define STATUS_HEATERS_X 5
  #else
    #if ENABLED(STATUS_COMBINE_HEATERS) && HAS_HEATED_BED && HOTENDS <= 4
      #define STATUS_HEATERS_X 5
    #else
      #define STATUS_HEATERS_X 8 // Like the included bitmaps
    #endif
  #endif
#endif

#if STATUS_HOTEND1_WIDTH

  //
  // Hotend images. A base hotend image and optional "ON" state image.
  //
  #ifndef STATUS_HOTEND_BITMAPS
    #define STATUS_HOTEND_BITMAPS 1
  #endif

  #ifndef STATUS_HOTEND2_WIDTH
    #define STATUS_HOTEND2_WIDTH STATUS_HOTEND1_WIDTH
  #endif
  #ifndef STATUS_HOTEND3_WIDTH
    #define STATUS_HOTEND3_WIDTH STATUS_HOTEND2_WIDTH
  #endif
  #ifndef STATUS_HOTEND4_WIDTH
    #define STATUS_HOTEND4_WIDTH STATUS_HOTEND3_WIDTH
  #endif
  #ifndef STATUS_HOTEND5_WIDTH
    #define STATUS_HOTEND5_WIDTH STATUS_HOTEND4_WIDTH
  #endif
  #ifndef STATUS_HOTEND6_WIDTH
    #define STATUS_HOTEND6_WIDTH STATUS_HOTEND5_WIDTH
  #endif

  constexpr uint8_t status_hotend_width[HOTENDS] = ARRAY_N(HOTENDS, STATUS_HOTEND1_WIDTH, STATUS_HOTEND2_WIDTH, STATUS_HOTEND3_WIDTH, STATUS_HOTEND4_WIDTH, STATUS_HOTEND5_WIDTH, STATUS_HOTEND6_WIDTH);
  #define STATUS_HOTEND_WIDTH(N) status_hotend_width[N]

  #ifndef STATUS_HOTEND1_BYTEWIDTH
    #define STATUS_HOTEND1_BYTEWIDTH BW(STATUS_HOTEND1_WIDTH)
  #endif
  #ifndef STATUS_HOTEND2_BYTEWIDTH
    #define STATUS_HOTEND2_BYTEWIDTH BW(STATUS_HOTEND2_WIDTH)
  #endif
  #ifndef STATUS_HOTEND3_BYTEWIDTH
    #define STATUS_HOTEND3_BYTEWIDTH BW(STATUS_HOTEND3_WIDTH)
  #endif
  #ifndef STATUS_HOTEND4_BYTEWIDTH
    #define STATUS_HOTEND4_BYTEWIDTH BW(STATUS_HOTEND4_WIDTH)
  #endif
  #ifndef STATUS_HOTEND5_BYTEWIDTH
    #define STATUS_HOTEND5_BYTEWIDTH BW(STATUS_HOTEND5_WIDTH)
  #endif
  #ifndef STATUS_HOTEND6_BYTEWIDTH
    #define STATUS_HOTEND6_BYTEWIDTH BW(STATUS_HOTEND6_WIDTH)
  #endif

  constexpr uint8_t status_hotend_bytewidth[HOTENDS] = ARRAY_N(HOTENDS, STATUS_HOTEND1_BYTEWIDTH, STATUS_HOTEND2_BYTEWIDTH, STATUS_HOTEND3_BYTEWIDTH, STATUS_HOTEND4_BYTEWIDTH, STATUS_HOTEND5_BYTEWIDTH, STATUS_HOTEND6_BYTEWIDTH);
  #define STATUS_HOTEND_BYTEWIDTH(N) status_hotend_bytewidth[N]

  #ifndef STATUS_HOTEND1_X
    #define STATUS_HOTEND1_X STATUS_HEATERS_X
  #endif
  #ifndef STATUS_HOTEND2_X
    #define STATUS_HOTEND2_X STATUS_HOTEND1_X + STATUS_HEATERS_XSPACE
  #endif

  #if HOTENDS > 2
    #ifndef STATUS_HOTEND3_X
      #define STATUS_HOTEND3_X STATUS_HOTEND2_X + STATUS_HEATERS_XSPACE
    #endif
    #ifndef STATUS_HOTEND4_X
      #define STATUS_HOTEND4_X STATUS_HOTEND3_X + STATUS_HEATERS_XSPACE
    #endif
    #ifndef STATUS_HOTEND5_X
      #define STATUS_HOTEND5_X STATUS_HOTEND4_X + STATUS_HEATERS_XSPACE
    #endif
    #ifndef STATUS_HOTEND6_X
      #define STATUS_HOTEND6_X STATUS_HOTEND5_X + STATUS_HEATERS_XSPACE
    #endif

    constexpr uint8_t status_hotend_x[HOTENDS] = ARRAY_N(HOTENDS, STATUS_HOTEND1_X, STATUS_HOTEND2_X, STATUS_HOTEND3_X, STATUS_HOTEND4_X, STATUS_HOTEND5_X, STATUS_HOTEND6_X);
    #define STATUS_HOTEND_X(N) status_hotend_x[N]
  #elif HOTENDS > 1
    #define STATUS_HOTEND_X(N) ((N) ? STATUS_HOTEND2_X : STATUS_HOTEND1_X)
  #else
    #define STATUS_HOTEND_X(N) STATUS_HOTEND1_X
  #endif

  #ifndef STATUS_HOTEND_TEXT_X
    #ifdef STATUS_HOTEND1_TEXT_X
      #ifndef STATUS_HOTEND2_TEXT_X
        #define STATUS_HOTEND2_TEXT_X STATUS_HOTEND1_TEXT_X + STATUS_HEATERS_XSPACE
      #endif
      #ifndef STATUS_HOTEND3_TEXT_X
        #define STATUS_HOTEND3_TEXT_X STATUS_HOTEND2_TEXT_X + STATUS_HEATERS_XSPACE
      #endif
      #ifndef STATUS_HOTEND4_TEXT_X
        #define STATUS_HOTEND4_TEXT_X STATUS_HOTEND3_TEXT_X + STATUS_HEATERS_XSPACE
      #endif
      #ifndef STATUS_HOTEND5_TEXT_X
        #define STATUS_HOTEND5_TEXT_X STATUS_HOTEND4_TEXT_X + STATUS_HEATERS_XSPACE
      #endif
      #ifndef STATUS_HOTEND6_TEXT_X
        #define STATUS_HOTEND6_TEXT_X STATUS_HOTEND5_TEXT_X + STATUS_HEATERS_XSPACE
      #endif
      constexpr uint8_t status_hotend_text_x[] = ARRAY_N(HOTENDS, STATUS_HOTEND1_TEXT_X, STATUS_HOTEND2_TEXT_X, STATUS_HOTEND3_TEXT_X, STATUS_HOTEND4_TEXT_X, STATUS_HOTEND5_TEXT_X, STATUS_HOTEND6_TEXT_X);
      #define STATUS_HOTEND_TEXT_X(N) status_hotend_text_x[N]
    #else
      #define STATUS_HOTEND_TEXT_X(N) (STATUS_HOTEND1_X + 6 + (N) * (STATUS_HEATERS_XSPACE))
    #endif
  #endif

  #if STATUS_HOTEND_BITMAPS > 1 && DISABLED(STATUS_HOTEND_NUMBERLESS)
    #define TEST_BITMAP_OFF status_hotend1_a_bmp
    #define TEST_BITMAP_ON  status_hotend1_b_bmp
  #else
    #define TEST_BITMAP_OFF status_hotend_a_bmp
    #define TEST_BITMAP_ON  status_hotend_b_bmp
  #endif
  #ifndef STATUS_HEATERS_HEIGHT
    #define STATUS_HEATERS_HEIGHT (sizeof(TEST_BITMAP_OFF) / (STATUS_HOTEND1_BYTEWIDTH))
  #endif
  #ifndef STATUS_HEATERS_Y
    #define STATUS_HEATERS_Y (20 - (STATUS_HEATERS_HEIGHT))
  #endif

  #define HOTEND0_BITMAP_SIZE (STATUS_HOTEND1_BYTEWIDTH) * (STATUS_HEATERS_HEIGHT)
  static_assert(
    sizeof(TEST_BITMAP_OFF) == HOTEND0_BITMAP_SIZE,
    "Status hotend bitmap (" STRINGIFY(TEST_BITMAP_OFF) ") dimensions don't match data."
  );
  #ifdef STATUS_HOTEND_ANIM
    static_assert(
      sizeof(TEST_BITMAP_ON) == HOTEND0_BITMAP_SIZE,
      "Status hotend bitmaps (" STRINGIFY(TEST_BITMAP_OFF) " and " STRINGIFY(TEST_BITMAP_ON) ") dimensions don't match."
    );
  #endif

#elif STATUS_HEATERS_WIDTH

  #ifndef STATUS_HEATERS_XSPACE
    #define STATUS_HEATERS_XSPACE 24
  #endif
  #ifndef STATUS_HOTEND_WIDTH
    #define STATUS_HOTEND_WIDTH(N) 10
  #endif
  #ifndef STATUS_HOTEND_X
    #define STATUS_HOTEND_X(N) (STATUS_HEATERS_X + 2 + (N) * (STATUS_HEATERS_XSPACE))
  #endif
  #ifndef STATUS_HOTEND_TEXT_X
    #if HOTENDS == 4 && STATUS_LOGO_WIDTH
      #define STATUS_HOTEND_TEXT_X(N) (STATUS_HEATERS_X + 6 + (N) * (STATUS_HEATERS_XSPACE))
    #else
      #define STATUS_HOTEND_TEXT_X(N) (STATUS_HEATERS_X + 6 + (N) * (STATUS_HEATERS_XSPACE))
    #endif
  #endif
  #ifndef STATUS_HEATERS_BYTEWIDTH
    #define STATUS_HEATERS_BYTEWIDTH BW(STATUS_HEATERS_WIDTH)
  #endif
  #ifndef STATUS_HEATERS_HEIGHT
    #define STATUS_HEATERS_HEIGHT (sizeof(status_heaters_bmp) / (STATUS_HEATERS_BYTEWIDTH))
  #endif
  #ifndef STATUS_HEATERS_Y
    #define STATUS_HEATERS_Y (20 - (STATUS_HEATERS_HEIGHT))
  #endif

  static_assert(
    sizeof(status_heaters_bmp) == (STATUS_HEATERS_BYTEWIDTH) * (STATUS_HEATERS_HEIGHT),
    "Status heaters bitmap (status_heaters_bmp) dimensions don't match data."
  );

#else // HOTENDS == 0

  #define STATUS_HOTEND_TEXT_X(N) 0
  #define STATUS_HEATERS_Y 0

#endif

//
// Cutter Bitmap Properties
//
#ifndef STATUS_CUTTER_BYTEWIDTH
  #define STATUS_CUTTER_BYTEWIDTH BW(STATUS_CUTTER_WIDTH)
#endif
#if STATUS_CUTTER_WIDTH

  #ifndef STATUS_CUTTER_X
    #define STATUS_CUTTER_X (LCD_PIXEL_WIDTH - (STATUS_CUTTER_BYTEWIDTH + STATUS_CUTTER_BYTEWIDTH) * 8)
  #endif

  #ifndef STATUS_CUTTER_HEIGHT
    #ifdef STATUS_CUTTER_ANIM
      #define STATUS_CUTTER_HEIGHT(S) ((S) ? sizeof(status_cutter_on_bmp) / (STATUS_CUTTER_BYTEWIDTH) : sizeof(status_cutter_bmp) / (STATUS_CUTTER_BYTEWIDTH))
    #else
      #define STATUS_CUTTER_HEIGHT(S) (sizeof(status_cutter_bmp) / (STATUS_CUTTER_BYTEWIDTH))
    #endif
  #endif

  #ifndef STATUS_CUTTER_Y
    #define STATUS_CUTTER_Y(S) 4
  #endif

  #ifndef STATUS_CUTTER_TEXT_X
    #define STATUS_CUTTER_TEXT_X (STATUS_CUTTER_X -1)
  #endif

  #ifndef STATUS_CUTTER_TEXT_Y
    #define STATUS_CUTTER_TEXT_Y 28
  #endif

  static_assert(
    sizeof(status_cutter_bmp) == (STATUS_CUTTER_BYTEWIDTH) * (STATUS_CUTTER_HEIGHT(0)),
    "Status cutter bitmap (status_cutter_bmp) dimensions don't match data."
  );
  #ifdef STATUS_CUTTER_ANIM
    static_assert(
      sizeof(status_cutter_on_bmp) == (STATUS_CUTTER_BYTEWIDTH) * (STATUS_CUTTER_HEIGHT(1)),
      "Status cutter bitmap (status_cutter_on_bmp) dimensions don't match data."
    );
  #endif

#endif

//
// Chamber Bitmap Properties
//
#ifndef STATUS_CHAMBER_BYTEWIDTH
  #define STATUS_CHAMBER_BYTEWIDTH BW(STATUS_CHAMBER_WIDTH)
#endif
#if STATUS_CHAMBER_WIDTH

  #ifndef STATUS_CHAMBER_X
    #define STATUS_CHAMBER_X (LCD_PIXEL_WIDTH - (STATUS_CHAMBER_BYTEWIDTH + STATUS_FAN_BYTEWIDTH) * 8)
  #endif

  #ifndef STATUS_CHAMBER_HEIGHT
    #ifdef STATUS_CHAMBER_ANIM
      #define STATUS_CHAMBER_HEIGHT(S) ((S) ? sizeof(status_chamber_on_bmp) / (STATUS_CHAMBER_BYTEWIDTH) : sizeof(status_chamber_bmp) / (STATUS_CHAMBER_BYTEWIDTH))
    #else
      #define STATUS_CHAMBER_HEIGHT(S) (sizeof(status_chamber_bmp) / (STATUS_CHAMBER_BYTEWIDTH))
    #endif
  #endif

  #ifndef STATUS_CHAMBER_Y
    #define STATUS_CHAMBER_Y(S) (20 - STATUS_CHAMBER_HEIGHT(S))
  #endif

  #ifndef STATUS_CHAMBER_TEXT_X
    #define STATUS_CHAMBER_TEXT_X (STATUS_CHAMBER_X + 11)
  #endif

  static_assert(
    sizeof(status_chamber_bmp) == (STATUS_CHAMBER_BYTEWIDTH) * (STATUS_CHAMBER_HEIGHT(0)),
    "Status chamber bitmap (status_chamber_bmp) dimensions don't match data."
  );
  #ifdef STATUS_CHAMBER_ANIM
    static_assert(
      sizeof(status_chamber_on_bmp) == (STATUS_CHAMBER_BYTEWIDTH) * (STATUS_CHAMBER_HEIGHT(1)),
      "Status chamber bitmap (status_chamber_on_bmp) dimensions don't match data."
    );
  #endif

#endif

//
// Bed Bitmap Properties
//
#ifndef STATUS_BED_BYTEWIDTH
  #define STATUS_BED_BYTEWIDTH BW(STATUS_BED_WIDTH)
#endif
#if STATUS_BED_WIDTH && !STATUS_HEATERS_WIDTH

  #ifndef STATUS_BED_X
    #define STATUS_BED_X (LCD_PIXEL_WIDTH - (STATUS_CHAMBER_BYTEWIDTH + STATUS_FAN_BYTEWIDTH + STATUS_BED_BYTEWIDTH) * 8)
  #endif

  #ifndef STATUS_BED_HEIGHT
    #ifdef STATUS_BED_ANIM
      #define STATUS_BED_HEIGHT(S) ((S) ? sizeof(status_bed_on_bmp) / (STATUS_BED_BYTEWIDTH) : sizeof(status_bed_bmp) / (STATUS_BED_BYTEWIDTH))
    #else
      #define STATUS_BED_HEIGHT(S) (sizeof(status_bed_bmp) / (STATUS_BED_BYTEWIDTH))
    #endif
  #endif

  #ifndef STATUS_BED_Y
    #define STATUS_BED_Y(S) (20 - STATUS_BED_HEIGHT(S))
  #endif

  #ifndef STATUS_BED_TEXT_X
    #define STATUS_BED_TEXT_X (STATUS_BED_X + 9)
  #endif

  static_assert(
    sizeof(status_bed_bmp) == (STATUS_BED_BYTEWIDTH) * (STATUS_BED_HEIGHT(0)),
    "Status bed bitmap (status_bed_bmp) dimensions don't match data."
  );
  #ifdef STATUS_BED_ANIM
    static_assert(
      sizeof(status_bed_on_bmp) == (STATUS_BED_BYTEWIDTH) * (STATUS_BED_HEIGHT(1)),
      "Status bed bitmap (status_bed_on_bmp) dimensions don't match data."
    );
  #endif
#endif

//
// Fan Bitmap Properties
//
#ifndef STATUS_FAN_BYTEWIDTH
  #define STATUS_FAN_BYTEWIDTH BW(STATUS_FAN_WIDTH)
#endif
#if STATUS_FAN_FRAMES
  #ifndef STATUS_FAN_X
    #define STATUS_FAN_X (LCD_PIXEL_WIDTH - (STATUS_FAN_BYTEWIDTH) * 8)
  #endif
  #ifndef STATUS_FAN_Y
    #define STATUS_FAN_Y 1
  #endif
  #ifndef STATUS_FAN_TEXT_X
    #define STATUS_FAN_TEXT_X (STATUS_FAN_X - 1)
  #endif
  #ifndef STATUS_FAN_TEXT_Y
    #define STATUS_FAN_TEXT_Y 28
  #endif
  #ifndef STATUS_FAN_HEIGHT
    #define STATUS_FAN_HEIGHT (sizeof(status_fan0_bmp) / (STATUS_FAN_BYTEWIDTH))
  #endif
  #define FAN_BMP_SIZE (STATUS_FAN_BYTEWIDTH) * (STATUS_FAN_HEIGHT)
  static_assert(sizeof(status_fan0_bmp) == FAN_BMP_SIZE, "Status fan bitmap (status_fan0_bmp) dimensions don't match data.");
  #if STATUS_FAN_FRAMES > 1
    static_assert(sizeof(status_fan1_bmp) == FAN_BMP_SIZE, "Status fan bitmap (status_fan1_bmp) dimensions don't match data.");
    #if STATUS_FAN_FRAMES > 2
      static_assert(sizeof(status_fan2_bmp) == FAN_BMP_SIZE, "Status fan bitmap (status_fan2_bmp) dimensions don't match data.");
      #if STATUS_FAN_FRAMES > 3
        static_assert(sizeof(status_fan3_bmp) == FAN_BMP_SIZE, "Status fan bitmap (status_fan3_bmp) dimensions don't match data.");
      #endif
    #endif
  #endif
#endif

#define DO_DRAW_LOGO (STATUS_LOGO_WIDTH && ENABLED(CUSTOM_STATUS_SCREEN_IMAGE))
#define DO_DRAW_HOTENDS (HOTENDS > 0)
#define DO_DRAW_BED (HAS_HEATED_BED && HOTENDS <= 4)
#define DO_DRAW_CUTTER (HAS_CUTTER && !DO_DRAW_BED)
#define DO_DRAW_CHAMBER (HAS_TEMP_CHAMBER && STATUS_CHAMBER_WIDTH && HOTENDS <= 4)
#define DO_DRAW_FAN (HAS_FAN0 && STATUS_FAN_WIDTH && HOTENDS <= 4 && defined(STATUS_FAN_FRAMES))

#define ANIM_HOTEND (HOTENDS && ENABLED(STATUS_HOTEND_ANIM))
#define ANIM_BED (DO_DRAW_BED && ENABLED(STATUS_BED_ANIM))
#define ANIM_CHAMBER (DO_DRAW_CHAMBER && ENABLED(STATUS_CHAMBER_ANIM))
#define ANIM_CUTTER (DO_DRAW_CUTTER && ENABLED(STATUS_CUTTER_ANIM))

#define ANIM_HBCC (ANIM_HOTEND || ANIM_BED || ANIM_CHAMBER || ANIM_CUTTER)
